<?php
/**
 * Logs Page - Flowb0t DCI v2.0 Professional
 * Variables available: $data, $logsData, $logDomains, $processId, $currentPage
 * All text in ENGLISH
 */

$baseUrl = rtrim(dirname($_SERVER['SCRIPT_NAME']), '/\\');

// Get filter parameters
$filterStatus = $_GET['status'] ?? '';
$filterDomain = $_GET['domain'] ?? '';
$filterErrorType = $_GET['error_type'] ?? '';
$filterSearch = $_GET['search'] ?? '';
$currentPageNum = (int)($_GET['p'] ?? 1);

// Get logs data
$logs = $logsData['logs'] ?? [];
$totalLogs = $logsData['total'] ?? 0;
$totalPages = $logsData['total_pages'] ?? 1;
$perPage = $logsData['per_page'] ?? 50;

// Status counts for quick filters
$statusCounts = [
    'success' => 0,
    'warning' => 0,
    'error' => 0,
    'info' => 0,
];
// Note: These would ideally come from backend, but we approximate from current page
foreach ($logs as $log) {
    $class = $log['class'] ?? 'info';
    if (isset($statusCounts[$class])) {
        $statusCounts[$class]++;
    }
}

// Error types for filter dropdown
$errorTypes = [
    'timeout' => 'Timeout',
    'http_429' => 'HTTP 429 (Rate Limit)',
    'http_404' => 'HTTP 404 (Not Found)',
    'http_403' => 'HTTP 403 (Forbidden)',
    'http_5xx' => 'Server Error (5xx)',
    'connection' => 'Connection Error',
    'metadata' => 'Metadata Error',
    'other' => 'Other',
];
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Logs - <?= htmlspecialchars($processId) ?></title>
    <link rel="stylesheet" href="<?= $baseUrl ?>/assets/css/style.css">
    <style>
        body {
            background: linear-gradient(135deg, #0F172A 0%, #1E293B 50%, #0F172A 100%);
            min-height: 100vh;
        }
    </style>
</head>
<body>
    <?php include __DIR__ . '/partials/navigation.php'; ?>

    <div class="dashboard compact">
        <!-- ============ HEADER ============ -->
        <div class="page-header glass-card">
            <h1>Processing Logs</h1>
            <p>Complete log history with search and filtering</p>
        </div>

        <!-- ============ FILTERS PANEL ============ -->
        <div class="filters-panel glass-card">
            <form method="GET" class="filters-form" id="logsFilterForm">
                <input type="hidden" name="process_id" value="<?= htmlspecialchars($processId) ?>">
                <input type="hidden" name="page" value="logs">

                <div class="filter-group">
                    <label for="filterStatus">Status</label>
                    <select name="status" id="filterStatus" class="filter-select">
                        <option value="">All Status</option>
                        <option value="success" <?= $filterStatus === 'success' ? 'selected' : '' ?>>Success</option>
                        <option value="warning" <?= $filterStatus === 'warning' ? 'selected' : '' ?>>Warning</option>
                        <option value="error" <?= $filterStatus === 'error' ? 'selected' : '' ?>>Error</option>
                    </select>
                </div>

                <div class="filter-group">
                    <label for="filterDomain">Domain</label>
                    <select name="domain" id="filterDomain" class="filter-select">
                        <option value="">All Domains</option>
                        <?php foreach ($logDomains as $domain): ?>
                        <option value="<?= htmlspecialchars($domain) ?>" <?= $filterDomain === $domain ? 'selected' : '' ?>>
                            <?= htmlspecialchars($domain) ?>
                        </option>
                        <?php endforeach; ?>
                    </select>
                </div>

                <div class="filter-group">
                    <label for="filterErrorType">Error Type</label>
                    <select name="error_type" id="filterErrorType" class="filter-select">
                        <option value="">All Types</option>
                        <?php foreach ($errorTypes as $key => $label): ?>
                        <option value="<?= $key ?>" <?= $filterErrorType === $key ? 'selected' : '' ?>>
                            <?= $label ?>
                        </option>
                        <?php endforeach; ?>
                    </select>
                </div>

                <div class="filter-group search-group">
                    <label for="filterSearch">Search</label>
                    <input type="text" name="search" id="filterSearch" class="search-input"
                           placeholder="Search URL or message..."
                           value="<?= htmlspecialchars($filterSearch) ?>">
                </div>

                <div class="filter-actions">
                    <button type="submit" class="btn btn-primary">Apply Filters</button>
                    <a href="?process_id=<?= urlencode($processId) ?>&page=logs" class="btn btn-secondary">Clear</a>
                </div>
            </form>
        </div>

        <!-- ============ QUICK FILTERS ============ -->
        <div class="quick-filters glass-card">
            <a href="?process_id=<?= urlencode($processId) ?>&page=logs"
               class="quick-filter <?= empty($filterStatus) ? 'active' : '' ?>">
                All <span class="count"><?= number_format($totalLogs) ?></span>
            </a>
            <a href="?process_id=<?= urlencode($processId) ?>&page=logs&status=success"
               class="quick-filter success <?= $filterStatus === 'success' ? 'active' : '' ?>">
                Success
            </a>
            <a href="?process_id=<?= urlencode($processId) ?>&page=logs&status=warning"
               class="quick-filter warning <?= $filterStatus === 'warning' ? 'active' : '' ?>">
                Warnings
            </a>
            <a href="?process_id=<?= urlencode($processId) ?>&page=logs&status=error"
               class="quick-filter error <?= $filterStatus === 'error' ? 'active' : '' ?>">
                Errors
            </a>
        </div>

        <!-- ============ LOGS TABLE ============ -->
        <div class="table-container glass-card">
            <div class="table-header">
                <div class="table-title">
                    <h3>Log Entries</h3>
                    <span class="table-count">
                        Showing <?= (($currentPageNum - 1) * $perPage) + 1 ?>-<?= min($currentPageNum * $perPage, $totalLogs) ?>
                        of <?= number_format($totalLogs) ?>
                    </span>
                </div>
                <div class="table-actions">
                    <div class="export-buttons">
                        <a href="?process_id=<?= urlencode($processId) ?>&page=api&action=export&type=logs&format=csv" class="export-btn csv">Export CSV</a>
                        <a href="?process_id=<?= urlencode($processId) ?>&page=api&action=export&type=logs&format=json" class="export-btn json">Export JSON</a>
                    </div>
                </div>
            </div>

            <div class="table-responsive">
                <table class="data-table logs-table" id="logsTable">
                    <thead>
                        <tr>
                            <th style="width: 70px;">Time</th>
                            <th style="width: 140px;">Domain</th>
                            <th>URL</th>
                            <th style="width: 70px;">Status</th>
                            <th style="width: 50px;">Phase</th>
                            <th style="width: 60px;">HTTP</th>
                            <th style="width: 70px;">Response</th>
                            <th>Message</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if (empty($logs)): ?>
                        <tr>
                            <td colspan="8" class="text-center">No log entries found matching your filters.</td>
                        </tr>
                        <?php else: ?>
                        <?php foreach ($logs as $log):
                            $class = $log['class'] ?? 'info';
                            $statusIcon = match($class) {
                                'success' => '&#10004;',
                                'warning' => '&#9888;',
                                'error' => '&#10060;',
                                default => '&#8505;',
                            };
                            $statusLabel = match($class) {
                                'success' => 'OK',
                                'warning' => 'Skip',
                                'error' => 'Err',
                                default => 'Info',
                            };
                            $timestamp = $log['timestamp'] ?? '';
                            $time = $timestamp ? date('H:i:s', strtotime($timestamp)) : '';
                            $responseTime = $log['response_time'] ?? 0;
                            $responseClass = $responseTime > 5 ? 'slow' : ($responseTime > 2 ? 'medium' : '');
                        ?>
                        <tr class="log-row <?= $class ?>">
                            <td class="log-time"><?= $time ?></td>
                            <td class="log-domain" title="<?= htmlspecialchars($log['domain'] ?? '') ?>">
                                <?= htmlspecialchars(substr($log['domain'] ?? '', 0, 18)) ?><?= strlen($log['domain'] ?? '') > 18 ? '...' : '' ?>
                            </td>
                            <td class="log-url" title="<?= htmlspecialchars($log['url'] ?? '') ?>">
                                <span class="url-text"><?= htmlspecialchars(substr($log['url'] ?? '', 0, 50)) ?><?= strlen($log['url'] ?? '') > 50 ? '...' : '' ?></span>
                            </td>
                            <td class="log-status">
                                <span class="status-badge <?= $class ?>"><?= $statusIcon ?> <?= $statusLabel ?></span>
                            </td>
                            <td class="log-phase text-center"><?= $log['phase'] ?? '-' ?></td>
                            <td class="log-http text-center">
                                <?php if (!empty($log['http_code']) && $log['http_code'] > 0): ?>
                                <span class="http-badge <?= $log['http_code'] >= 400 ? 'error' : 'success' ?>"><?= $log['http_code'] ?></span>
                                <?php else: ?>
                                -
                                <?php endif; ?>
                            </td>
                            <td class="log-response text-center <?= $responseClass ?>">
                                <?php if ($responseTime > 0): ?>
                                <?= number_format($responseTime, 2) ?>s
                                <?php else: ?>
                                -
                                <?php endif; ?>
                            </td>
                            <td class="log-message" title="<?= htmlspecialchars($log['message'] ?? '') ?>">
                                <?= htmlspecialchars(substr($log['message'] ?? '', 0, 40)) ?><?= strlen($log['message'] ?? '') > 40 ? '...' : '' ?>
                            </td>
                        </tr>
                        <?php endforeach; ?>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>

            <!-- ============ PAGINATION ============ -->
            <?php if ($totalPages > 1): ?>
            <div class="pagination">
                <?php
                $baseQueryParams = [
                    'process_id' => $processId,
                    'page' => 'logs',
                ];
                if ($filterStatus) $baseQueryParams['status'] = $filterStatus;
                if ($filterDomain) $baseQueryParams['domain'] = $filterDomain;
                if ($filterErrorType) $baseQueryParams['error_type'] = $filterErrorType;
                if ($filterSearch) $baseQueryParams['search'] = $filterSearch;
                ?>

                <?php if ($currentPageNum > 1): ?>
                <a href="?<?= http_build_query(array_merge($baseQueryParams, ['p' => $currentPageNum - 1])) ?>" class="page-link">&#8592; Prev</a>
                <?php endif; ?>

                <?php
                $startPage = max(1, $currentPageNum - 2);
                $endPage = min($totalPages, $currentPageNum + 2);

                if ($startPage > 1): ?>
                <a href="?<?= http_build_query(array_merge($baseQueryParams, ['p' => 1])) ?>" class="page-link">1</a>
                <?php if ($startPage > 2): ?>
                <span class="page-ellipsis">...</span>
                <?php endif; ?>
                <?php endif; ?>

                <?php for ($i = $startPage; $i <= $endPage; $i++): ?>
                <a href="?<?= http_build_query(array_merge($baseQueryParams, ['p' => $i])) ?>"
                   class="page-link <?= $i === $currentPageNum ? 'active' : '' ?>"><?= $i ?></a>
                <?php endfor; ?>

                <?php if ($endPage < $totalPages): ?>
                <?php if ($endPage < $totalPages - 1): ?>
                <span class="page-ellipsis">...</span>
                <?php endif; ?>
                <a href="?<?= http_build_query(array_merge($baseQueryParams, ['p' => $totalPages])) ?>" class="page-link"><?= $totalPages ?></a>
                <?php endif; ?>

                <?php if ($currentPageNum < $totalPages): ?>
                <a href="?<?= http_build_query(array_merge($baseQueryParams, ['p' => $currentPageNum + 1])) ?>" class="page-link">Next &#8594;</a>
                <?php endif; ?>
            </div>
            <?php endif; ?>
        </div>

        <!-- ============ ERROR PATTERNS ANALYSIS ============ -->
        <?php
        // Quick error pattern analysis from current logs
        $errorPatterns = [];
        foreach ($logs as $log) {
            if (($log['class'] ?? '') === 'error' && !empty($log['error_type'])) {
                $errorPatterns[$log['error_type']] = ($errorPatterns[$log['error_type']] ?? 0) + 1;
            }
        }
        arsort($errorPatterns);
        ?>
        <?php if (!empty($errorPatterns)): ?>
        <div class="patterns-card glass-card">
            <h3>Error Patterns (Current Page)</h3>
            <div class="patterns-grid">
                <?php foreach (array_slice($errorPatterns, 0, 4, true) as $type => $count):
                    $label = $errorTypes[$type] ?? ucfirst($type);
                ?>
                <div class="pattern-item">
                    <span class="pattern-type"><?= $label ?></span>
                    <span class="pattern-count"><?= $count ?></span>
                </div>
                <?php endforeach; ?>
            </div>
        </div>
        <?php endif; ?>
    </div>

    <script src="<?= $baseUrl ?>/assets/js/app.js"></script>
    <script>
        window.processId = '<?= htmlspecialchars($processId) ?>';

        // Initialize logs page functionality
        document.addEventListener('DOMContentLoaded', function() {
            FlowbotDCI.initLogsPage();
        });
    </script>
</body>
</html>
